<?php
/**
 * Sets up theme defaults and registers support for various WordPress features.
 *
 * Note that this function is hooked into the after_setup_theme hook, which
 * runs before the init hook. The init hook is too late for some features, such
 * as indicating support for post thumbnails.
 */
// Theme version.
if ( !defined( 'OCOLUS' ) ) {
	define( 'OCOLUS', wp_get_theme()->get( 'Version' ) );
}
if ( !function_exists( 'ocolus_theme_setup' ) ) {
	function ocolus_theme_setup()
	{
		// Set the default content width.
		$GLOBALS['content_width'] = 525;
		/*
		 * Make theme available for translation.
		 * Translations can be filed at WordPress.org. See: https://translate.wordpress.org/projects/wp-themes/ocolus
		 * If you're building a theme based on Twenty Seventeen, use a find and replace
		 * to change 'ocolus' to the name of your theme in all the template files.
		 */
		load_theme_textdomain( 'ocolus', get_template_directory() . '/languages' );
		// Add default posts and comments RSS feed links to head.
		add_theme_support( 'automatic-feed-links' );
		/*
		 * Let WordPress manage the document title.
		 * By adding theme support, we declare that this theme does not use a
		 * hard-coded <title> tag in the document head, and expect WordPress to
		 * provide it for us.
		 */
		add_theme_support( 'title-tag' );
		/*
		 * Enable support for Post Thumbnails on posts and pages.
		 *
		 * @link https://developer.wordpress.org/themes/functionality/featured-images-post-thumbnails/
		 */
		add_theme_support( 'post-thumbnails' );
		/*
		 * Switch default core markup for search form, comment form, and comments
		 * to output valid HTML5.
		 */
		add_theme_support( 'html5', array(
				'search-form',
				'comment-form',
				'comment-list',
				'gallery',
				'caption',
				'widgets',
			)
		);
		// This theme uses wp_nav_menu() in two locations.
		register_nav_menus( array(
				'primary'  => esc_html__( 'Primary Menu', 'ocolus' ),
				'account'  => esc_html__( 'Account Menu', 'ocolus' ),
				'vertical' => esc_html__( 'Vertical Menu', 'ocolus' ),
			)
		);
		/*
		 * Enable support for Post Formats.
		 *
		 * See: https://codex.wordpress.org/Post_Formats
		 */
		add_theme_support( 'post-formats', array(
				'aside',
				'image',
				'video',
				'quote',
				'link',
				'gallery',
				'audio',
			)
		);
		// Add support for Block Styles.
		add_theme_support( 'wp-block-styles' );
		// Add support for full and wide align images.
		add_theme_support( 'align-wide' );
		// Add support for responsive embedded content.
		add_theme_support( 'responsive-embeds' );
		// Add theme support for selective refresh for widgets.
		add_theme_support( 'customize-selective-refresh-widgets' );
		/*Support woocommerce*/
		add_theme_support( 'woocommerce' );
		add_theme_support( 'wc-product-gallery-lightbox' );
		add_theme_support( 'wc-product-gallery-slider' );
		add_theme_support( 'wc-product-gallery-zoom' );
	}

	add_action( 'after_setup_theme', 'ocolus_theme_setup' );
}
/**
 * Register widget area.
 *
 * @link https://developer.wordpress.org/themes/functionality/sidebars/#registering-a-sidebar
 */
if ( !function_exists( 'ocolus_widgets_init' ) ) {
	function ocolus_widgets_init()
	{
		register_sidebar( array(
				'name'          => esc_html__( 'Widget Area', 'ocolus' ),
				'id'            => 'widget-area',
				'description'   => esc_html__( 'Add widgets here to appear in your blog sidebar.', 'ocolus' ),
				'before_widget' => '<div id="%1$s" class="widget %2$s">',
				'after_widget'  => '</div>',
				'before_title'  => '<h2 class="widget-title">',
				'after_title'   => '<span class="arrow"></span></h2>',
			)
		);
		register_sidebar( array(
				'name'          => esc_html__( 'Header Widget Area', 'ocolus' ),
				'id'            => 'header-widget-area',
				'description'   => esc_html__( 'Add widgets here to appear in your header sidebar.', 'ocolus' ),
				'before_widget' => '<div id="%1$s" class="widget %2$s">',
				'after_widget'  => '</div>',
				'before_title'  => '<h2 class="widget-title">',
				'after_title'   => '<span class="arrow"></span></h2>',
			)
		);
		register_sidebar( array(
				'name'          => esc_html__( 'Shop Widget Area', 'ocolus' ),
				'id'            => 'shop-widget-area',
				'description'   => esc_html__( 'Add widgets here to appear in your shop sidebar.', 'ocolus' ),
				'before_widget' => '<div id="%1$s" class="widget %2$s">',
				'after_widget'  => '</div>',
				'before_title'  => '<h2 class="widget-title">',
				'after_title'   => '<span class="arrow"></span></h2>',
			)
		);
		register_sidebar( array(
				'name'          => esc_html__( 'Product Widget Area', 'ocolus' ),
				'id'            => 'product-widget-area',
				'description'   => esc_html__( 'Add widgets here to appear in your product sidebar.', 'ocolus' ),
				'before_widget' => '<div id="%1$s" class="widget %2$s">',
				'after_widget'  => '</div>',
				'before_title'  => '<h2 class="widget-title">',
				'after_title'   => '<span class="arrow"></span></h2>',
			)
		);
		register_sidebar( array(
				'name'          => esc_html__( 'Vendor Widget Area', 'ocolus' ),
				'id'            => 'vendor-widget-area',
				'description'   => esc_html__( 'Add widgets here to appear in vendor sidebar.', 'ocolus' ),
				'before_widget' => '<div id="%1$s" class="widget %2$s">',
				'after_widget'  => '</div>',
				'before_title'  => '<h2 class="widget-title">',
				'after_title'   => '<span class="arrow"></span></h2>',
			)
		);
		$sidebars = ocolus_get_option( 'multi_sidebar' );
		if ( is_array( $sidebars ) && count( $sidebars ) > 0 ) {
			foreach ( $sidebars as $sidebar ) {
				if ( $sidebar && $sidebar != '' ) {
					register_sidebar( array(
							'name'          => $sidebar['add_sidebar'],
							'id'            => 'custom-sidebar-' . sanitize_key( $sidebar['add_sidebar'] ),
							'before_widget' => '<div id="%1$s" class="widget %2$s">',
							'after_widget'  => '</div>',
							'before_title'  => '<h2 class="widget-title">',
							'after_title'   => '<span class="arrow"></span></h2>',
						)
					);
				}
			}
		}
	}

	add_action( 'widgets_init', 'ocolus_widgets_init' );
}
/**
 * Custom Body Class.
 */
if ( !function_exists( 'ocolus_body_class' ) ) {
	function ocolus_body_class( $classes )
	{
		$my_theme            = wp_get_theme();
		$header_mobile       = ocolus_get_option( 'mobile_style', 'style-01' );
		$quickview_style     = ocolus_get_option( 'quickview_style', 'style-01' );
		$page_main_container = ocolus_theme_option_meta( '_custom_page_side_options', null, 'page_main_container', '' );
		$header_options      = ocolus_theme_option_meta( '_custom_metabox_theme_options', 'header_template', 'metabox_header_template', 'style-05' );
		if ( ocolus_is_mobile() ) {
			$classes[] = "ocolus_is_mobile";
			$classes[] = "mobile-{$header_mobile}";
		}
		$classes[] = $page_main_container;
		$classes[] = "quickview-{$quickview_style}";
		$classes[] = strtolower( $my_theme->get( 'Name' ) ) . "-" . $my_theme->get( 'Version' );
		$classes[] = 'header-' . $header_options;

		return $classes;
	}

	add_filter( 'body_class', 'ocolus_body_class' );
}
/**
 * Add preconnect for Google Fonts.
 *
 * @since Ocolus 1.0
 *
 * @param array  $urls URLs to print for resource hints.
 * @param string $relation_type The relation type the URLs are printed.
 *
 * @return array $urls           URLs to print for resource hints.
 */
if ( !function_exists( 'ocolus_resource_hints' ) ) {
	function ocolus_resource_hints( $urls, $relation_type )
	{
		if ( wp_style_is( 'ocolus-fonts', 'queue' ) && 'preconnect' === $relation_type ) {
			$urls[] = array(
				'href' => 'https://fonts.gstatic.com',
				'crossorigin',
			);
		}

		return $urls;
	}

	add_filter( 'wp_resource_hints', 'ocolus_resource_hints', 10, 2 );
}
/**
 * Handles JavaScript detection.
 *
 * Adds a `js` class to the root `<html>` element when JavaScript is detected.
 *
 * @since Ocolus 1.0
 */
if ( !function_exists( 'ocolus_javascript_detection' ) ) {
	function ocolus_javascript_detection()
	{
		echo "<script>(function(html){html.className = html.className.replace(/\bno-js\b/,'js')})(document.documentElement);</script>\n";
	}

	add_action( 'wp_head', 'ocolus_javascript_detection', 0 );
}
/**
 * Add a pingback url auto-discovery header for single posts, pages, or attachments.
 */
if ( !function_exists( 'ocolus_pingback_header' ) ) {
	function ocolus_pingback_header()
	{
		if ( is_singular() && pings_open() ) {
			echo '<link rel="pingback" href="', esc_url( get_bloginfo( 'pingback_url' ) ), '">';
		}
	}

	add_action( 'wp_head', 'ocolus_pingback_header' );
}
/**
 * Functions Mobile Detect.
 */
if (!class_exists('Mobile_Detect')) {
    require get_theme_file_path('framework/classes/mobile-detect.php');
}
/**
 * Enqueue scripts and styles.
 */
require get_parent_theme_file_path( '/framework/settings/enqueue.php' );
/**
 * Functions plugin load.
 */
require get_parent_theme_file_path( '/framework/settings/plugins-load.php' );
/**
 * Functions theme AJAX.
 */
require get_parent_theme_file_path( '/framework/classes/core-ajax.php' );
/**
 * Functions theme breadcrumbs.
 */
require get_parent_theme_file_path( '/framework/classes/breadcrumbs.php' );
/**
 * Functions theme helper.
 */
require get_parent_theme_file_path( '/framework/settings/helper.php' );
/**
 * Functions theme options.
 */
require get_parent_theme_file_path( '/framework/settings/theme-options.php' );
/**
 * Functions add inline style and script.
 */
require get_parent_theme_file_path( '/framework/settings/color-patterns.php' );
/**
 * Functions theme.
 */
require get_parent_theme_file_path( '/framework/theme-functions.php' );
/**
 * Functions blog.
 */
require get_parent_theme_file_path( '/framework/blog-functions.php' );
/**
 * Functions Visual Composer.
 */
if ( class_exists( 'Vc_Manager' ) ) {
	require get_parent_theme_file_path( '/framework/visual-composer.php' );
}
/**
 * Functions WooCommerce.
 */
if ( class_exists( 'WooCommerce' ) ) {
	require get_parent_theme_file_path( '/framework/woocommerce/template-hook.php' );
}
/**
 * Functions Import Data.
 */
if ( class_exists( 'Import_Sample_Data' ) ) {
	require get_parent_theme_file_path( '/framework/importer/importer.php' );
}