<?php
/**
 * Handle frontend scripts
 *
 * @package Ocolus/Classes
 * @version 1.0.0
 */
if (!defined('ABSPATH')) {
    exit;
}
/**
 * Frontend scripts class.
 */
if (!class_exists('Ocolus_Assets')) {
    class Ocolus_Assets
    {
        /**
         * Contains an array of script handles registered by Ocolus.
         *
         * @var array
         */
        private static $scripts = array();
        /**
         * Contains an array of script handles registered by Ocolus.
         *
         * @var array
         */
        private static $styles = array();
        /**
         * Contains an array of script handles localized by Ocolus.
         *
         * @var array
         */
        private static $wp_localize_scripts = array();
        private static $suffix              = '';

        /**
         * Hook in methods.
         */
        public static function init()
        {
            /* check for developer mode */
            self::$suffix = defined('SCRIPT_DEBUG') && SCRIPT_DEBUG ? '' : '.min';

            add_action('admin_enqueue_scripts', array(__CLASS__, 'admin_scripts'));
            add_action('wp_enqueue_scripts', array(__CLASS__, 'load_scripts'));
            add_action('wp_print_scripts', array(__CLASS__, 'localize_printed_scripts'), 5);
            add_action('wp_print_footer_scripts', array(__CLASS__, 'localize_printed_scripts'), 5);
            add_action('wp_footer', array(__CLASS__, 'dequeue_footer_scripts'));
        }

        public static function dequeue_footer_scripts()
        {
            if (ocolus_is_mobile() && class_exists('WooCommerce')) {
                if (!is_product()) {
                    wp_dequeue_script('prettyPhoto');
                    wp_dequeue_script('prettyPhoto-init');
                    wp_dequeue_style('woocommerce_prettyPhoto_css');
                    /* SINGLE PRODUCT */
                    wp_dequeue_script('flexslider');
                    wp_dequeue_script('zoom');
                    wp_dequeue_script('wc-single-product');
                    wp_dequeue_script('wc-add-to-cart-variation');
                    /* PHOTO SWIPE */
                    wp_dequeue_script('photoswipe-ui-default');
                    wp_dequeue_style('photoswipe-default-skin');
                    remove_action('wp_footer', 'woocommerce_photoswipe', 15);
                }
            }
        }

        /**
         * Get google fonts.
         *
         * @return string
         */
        public static function fonts_url()
        {
            $subsets           = [
                'ru_RU' => 'cyrillic',
                'bg_BG' => 'cyrillic',
                'he_IL' => 'hebrew',
                'el'    => 'greek',
                'vi'    => 'vietnamese',
                'uk'    => 'cyrillic',
                'cs_CZ' => 'latin-ext',
                'ro_RO' => 'latin-ext',
                'pl_PL' => 'latin-ext',
                'hr_HR' => 'latin-ext',
                'hu_HU' => 'latin-ext',
                'sk_SK' => 'latin-ext',
                'tr_TR' => 'latin-ext',
                'lt_LT' => 'latin-ext',
            ];
            $subsets           = apply_filters('ocolus_frontend_google_font_subsets', $subsets);
            $enable_typography = ocolus_get_option('enable_typography');
            $typography_group  = ocolus_get_option('typography_group');
            $font_families     = array();
            if ($enable_typography == 1 && !empty($typography_group)) {
                foreach ($typography_group as $typography) {
                    $font_families[] = str_replace(' ', '+', $typography['typography_font_family']['family']);
                }
            }
            $font_families[] = 'Poppins:300,300i,400,400i,500,500i,600,600i,700,700i';
            $font_families[] = 'Montserrat:400,400i,600,600i,700,700i';
            $font_families[] = 'Playfair Display:400,400i,700,700i';
            $font_families[] = 'Oswald:300,400,500,600';
            $font_families[] = 'Manrope:300,400,500,600,700';
            $query_args      = array(
                'family'  => implode('%7C', $font_families),
                'display' => 'swap',
            );
            $fonts_url       = add_query_arg($query_args, 'https://fonts.googleapis.com/css');

            $locale = get_locale();

            if (isset($subsets[$locale])) {
                $fonts_url .= '&subset='.$subsets[$locale];
            }

            return esc_url_raw($fonts_url);
        }

        /**
         * Get styles for the frontend.
         *
         * @return array
         */
        public static function get_styles()
        {
            $styles = array(
                'animate-css'  => array(
                    'src'     => get_theme_file_uri('/assets/css/animate.min.css'),
                    'deps'    => array(),
                    'version' => '3.7.0',
                    'media'   => 'all',
                    'has_rtl' => false,
                ),
                'ocolus-fonts' => array(
                    'src'     => self::fonts_url(),
                    'deps'    => array(),
                    'version' => OCOLUS,
                    'media'   => 'all',
                    'has_rtl' => false,
                ),
                'chosen'       => array(
                    'src'     => get_theme_file_uri('/assets/vendor/chosen/chosen.min.css'),
                    'deps'    => array(),
                    'version' => '1.8.7',
                    'media'   => 'all',
                    'has_rtl' => false,
                ),
                'growl'        => array(
                    'src'     => get_theme_file_uri('/assets/vendor/growl/growl.min.css'),
                    'deps'    => array(),
                    'version' => '1.3.5',
                    'media'   => 'all',
                    'has_rtl' => false,
                ),
                'bootstrap'    => array(
                    'src'     => get_theme_file_uri('/assets/vendor/bootstrap/bootstrap.min.css'),
                    'deps'    => array(),
                    'version' => '3.4.0',
                    'media'   => 'all',
                    'has_rtl' => false,
                ),
                'slick'        => array(
                    'src'     => get_theme_file_uri('/assets/vendor/slick/slick.min.css'),
                    'deps'    => array(),
                    'version' => OCOLUS,
                    'media'   => 'all',
                    'has_rtl' => false,
                ),
            );
            /* ENQUEUE BY OPTIONS */
            if (!ocolus_is_mobile()) {
                $header_options = ocolus_theme_option_meta(
                    '_custom_metabox_theme_options',
                    'header_template',
                    'metabox_header_template',
                    'style-05'
                );
                if ($header_options == 'style-25') {
                    $header_options = 'style-10';
                }
                $style_header = "/templates/header/style/header-{$header_options}.css";
                $style_min    = "/templates/header/style/header-{$header_options}.min.css";
                if (is_file(get_template_directory().$style_min)) {
                    $style_header = $style_min;
                }
                if (is_file(get_template_directory().$style_header)) {
                    $styles['ocolus-header-style'] = array(
                        'src'     => get_theme_file_uri($style_header),
                        'deps'    => array(),
                        'version' => OCOLUS,
                        'media'   => 'all',
                        'has_rtl' => false,
                    );
                }
            } else {
                $styles['ocolus-header-mobile'] = array(
                    'src'     => get_theme_file_uri('/assets/css/style-mobile.min.css'),
                    'deps'    => array(),
                    'version' => OCOLUS,
                    'media'   => 'all',
                    'has_rtl' => false,
                );
            }
            /* STYLE MAIN */
            $styles['ocolus-custom-css'] = array(
                'src'     => get_theme_file_uri('/assets/css/style'.self::$suffix.'.css'),
                'deps'    => array('font-awesome', 'icomoon'),
                'version' => OCOLUS,
                'media'   => 'all',
                'has_rtl' => false,
            );
            $styles['ocolus-main']       = array(
                'src'     => get_stylesheet_uri(),
                'deps'    => array(),
                'version' => OCOLUS,
                'media'   => 'all',
                'has_rtl' => false,
            );

            return apply_filters('ocolus_enqueue_styles', $styles);
        }

        /**
         * Register a script for use.
         *
         * @param  string  $handle  Name of the script. Should be unique.
         * @param  string  $path  Full URL of the script, or path of the script relative to the WordPress root directory.
         * @param  string[]  $deps  An array of registered script handles this script depends on.
         * @param  string  $version  String specifying script version number, if it has one, which is added to the URL as a query string for cache busting purposes. If version is set to false, a version number is automatically added equal to current installed WordPress version. If set to null, no version is added.
         * @param  boolean  $in_footer  Whether to enqueue the script before </body> instead of in the <head>. Default 'false'.
         *
         * @uses   wp_register_script()
         *
         */
        private static function register_script($handle, $path, $deps = array('jquery'), $version = OCOLUS, $in_footer = true)
        {
            self::$scripts[] = $handle;
            wp_register_script($handle, $path, $deps, $version, $in_footer);
        }

        /**
         * Register and enqueue a script for use.
         *
         * @param  string  $handle  Name of the script. Should be unique.
         * @param  string  $path  Full URL of the script, or path of the script relative to the WordPress root directory.
         * @param  string[]  $deps  An array of registered script handles this script depends on.
         * @param  string  $version  String specifying script version number, if it has one, which is added to the URL as a query string for cache busting purposes. If version is set to false, a version number is automatically added equal to current installed WordPress version. If set to null, no version is added.
         * @param  boolean  $in_footer  Whether to enqueue the script before </body> instead of in the <head>. Default 'false'.
         *
         * @uses   wp_enqueue_script()
         *
         */
        private static function enqueue_script($handle, $path = '', $deps = array('jquery'), $version = OCOLUS, $in_footer = true)
        {
            if (!in_array($handle, self::$scripts, true) && $path) {
                self::register_script($handle, $path, $deps, $version, $in_footer);
            }
            wp_enqueue_script($handle);
        }

        /**
         * Register a style for use.
         *
         * @param  string  $handle  Name of the stylesheet. Should be unique.
         * @param  string  $path  Full URL of the stylesheet, or path of the stylesheet relative to the WordPress root directory.
         * @param  string[]  $deps  An array of registered stylesheet handles this stylesheet depends on.
         * @param  string  $version  String specifying stylesheet version number, if it has one, which is added to the URL as a query string for cache busting purposes. If version is set to false, a version number is automatically added equal to current installed WordPress version. If set to null, no version is added.
         * @param  string  $media  The media for which this stylesheet has been defined. Accepts media types like 'all', 'print' and 'screen', or media queries like '(orientation: portrait)' and '(max-width: 640px)'.
         * @param  boolean  $has_rtl  If has RTL version to load too.
         *
         * @uses   wp_register_style()
         *
         */
        private static function register_style($handle, $path, $deps = array(), $version = OCOLUS, $media = 'all', $has_rtl = false)
        {
            self::$styles[] = $handle;
            wp_register_style($handle, $path, $deps, $version, $media);
            if ($has_rtl) {
                wp_style_add_data($handle, 'rtl', 'replace');
            }
        }

        /**
         * Register and enqueue a styles for use.
         *
         * @param  string  $handle  Name of the stylesheet. Should be unique.
         * @param  string  $path  Full URL of the stylesheet, or path of the stylesheet relative to the WordPress root directory.
         * @param  string[]  $deps  An array of registered stylesheet handles this stylesheet depends on.
         * @param  string  $version  String specifying stylesheet version number, if it has one, which is added to the URL as a query string for cache busting purposes. If version is set to false, a version number is automatically added equal to current installed WordPress version. If set to null, no version is added.
         * @param  string  $media  The media for which this stylesheet has been defined. Accepts media types like 'all', 'print' and 'screen', or media queries like '(orientation: portrait)' and '(max-width: 640px)'.
         * @param  boolean  $has_rtl  If has RTL version to load too.
         *
         * @uses   wp_enqueue_style()
         *
         */
        private static function enqueue_style($handle, $path = '', $deps = array(), $version = OCOLUS, $media = 'all', $has_rtl = false)
        {
            if (!in_array($handle, self::$styles, true) && $path) {
                self::register_style($handle, $path, $deps, $version, $media, $has_rtl);
            }
            wp_enqueue_style($handle);
        }

        /**
         * Register all Ocolus scripts.
         */
        private static function register_scripts()
        {
            $register_scripts = array(
                'ocolus-main'               => array(
                    'src'     => get_theme_file_uri('/assets/js/frontend'.self::$suffix.'.js'),
                    'deps'    => array('jquery', 'bootstrap', 'lazyload', 'chosen', 'slick', 'growl'),
                    'version' => OCOLUS,
                ),
                'ocolus-backend'            => array(
                    'src'     => get_theme_file_uri('/assets/js/ocolus-backend.js'),
                    'deps'    => array('jquery'),
                    'version' => OCOLUS,
                ),
                'ocolus-sticky'             => array(
                    'src'     => get_theme_file_uri('/assets/js/sticky'.self::$suffix.'.js'),
                    'deps'    => array('jquery'),
                    'version' => OCOLUS,
                ),
                'ocolus-sticky-add-to-cart' => array(
                    'src'     => get_theme_file_uri('/assets/js/sticky-add-to-cart'.self::$suffix.'.js'),
                    'deps'    => array('jquery'),
                    'version' => OCOLUS,
                ),
                /* https://harvesthq.github.io/chosen/ */
                'chosen'                    => array(
                    'src'     => get_theme_file_uri('/assets/vendor/chosen/chosen.min.js'),
                    'deps'    => array('jquery'),
                    'version' => '1.8.7',
                ),
                /* http://jquery.eisbehr.de/lazy */
                'lazyload'                  => array(
                    'src'     => get_theme_file_uri('/assets/vendor/lazyload/lazyload.min.js'),
                    'deps'    => array('jquery'),
                    'version' => '1.7.10',
                ),
                /* http://hilios.github.io/jQuery.countdown/documentation.html */
                'countdown'                 => array(
                    'src'     => get_theme_file_uri('/assets/vendor/countdown/countdown.min.js'),
                    'deps'    => array('jquery'),
                    'version' => '2.2.0',
                ),
                'ocolus-countdown'          => array(
                    'src'     => get_theme_file_uri('/assets/js/countdown'.self::$suffix.'.js'),
                    'deps'    => array('countdown'),
                    'version' => OCOLUS,
                ),
                /* https://ksylvest.github.io/jquery-growl/ */
                'growl'                     => array(
                    'src'     => get_theme_file_uri('/assets/vendor/growl/growl.min.js'),
                    'deps'    => array('jquery'),
                    'version' => '1.3.5',
                ),
                /* https://getbootstrap.com/ */
                'bootstrap'                 => array(
                    'src'     => get_theme_file_uri('/assets/vendor/bootstrap/bootstrap.min.js'),
                    'deps'    => array('jquery'),
                    'version' => '3.4.0',
                ),
                /* http://kenwheeler.github.io/slick/ */
                'slick'                     => array(
                    'src'     => get_theme_file_uri('/assets/vendor/slick/slick.min.js'),
                    'deps'    => array('jquery'),
                    'version' => OCOLUS,
                ),
                /* https://isotope.metafizzy.co/ */
                'ocolus-isotope'            => array(
                    'src'     => get_theme_file_uri('/assets/vendor/isotope/isotope.pkgd.min.js'),
                    'deps'    => array('jquery'),
                    'version' => '3.0.6',
                ),
                'packery-mode'              => array(
                    'src'     => get_theme_file_uri('/assets/vendor/isotope/packery-mode.pkgd.min.js'),
                    'deps'    => array('jquery'),
                    'version' => '2.0.1',
                ),
                'ocolus-grid-isotope'       => array(
                    'src'     => get_theme_file_uri('/assets/js/isotope'.self::$suffix.'.js'),
                    'deps'    => array('imagesloaded', 'ocolus-isotope', 'packery-mode'),
                    'version' => OCOLUS,
                ),
                /* http://dimsemenov.com/plugins/magnific-popup/ */
                'magnific-popup'            => array(
                    'src'     => get_theme_file_uri('/assets/vendor/magnific-popup/magnific-popup.min.js'),
                    'deps'    => array('jquery'),
                    'version' => '1.1.0',
                ),
                'ocolus-magnific-popup'     => array(
                    'src'     => get_theme_file_uri('/assets/js/magnific-popup'.self::$suffix.'.js'),
                    'deps'    => array('magnific-popup'),
                    'version' => OCOLUS,
                ),
                /* https://github.com/alvarotrigo/fullPage.js */
                'fullpage'                  => array(
                    'src'     => get_theme_file_uri('/assets/vendor/fullpage/fullpage.min.js'),
                    'deps'    => array('jquery'),
                    'version' => '3.0.4',
                ),
                /* https://github.com/gromo/jquery.scrollbar/ */
                'scrollbar'                 => array(
                    'src'     => get_theme_file_uri('/assets/vendor/scrollbar/scrollbar.min.js'),
                    'deps'    => array('jquery'),
                    'version' => '0.2.10',
                ),
            );
            $google_map_api   = ocolus_get_option('google_map_api', '');
            if ($google_map_api != '') {
                $map_api = add_query_arg(
                    array(
                        'key'       => $google_map_api,
                        'libraries' => 'places',
                    ),
                    'https://maps.googleapis.com/maps/api/js'
                );
                /* init map */
                $register_scripts['ocolus-maps-api'] = array(
                    'src'     => esc_url_raw($map_api),
                    'deps'    => array('jquery'),
                    'version' => OCOLUS,
                );
            }
            foreach ($register_scripts as $name => $props) {
                self::register_script($name, $props['src'], $props['deps'], $props['version']);
            }
        }

        /**
         * Register all Ocolus styles.
         */
        private static function register_styles()
        {
            $register_styles = array(
                'ocolus-blog-single' => array(
                    'src'     => get_theme_file_uri('/assets/css/blog-single.min.css'),
                    'deps'    => array(),
                    'version' => OCOLUS,
                    'has_rtl' => false,
                ),
                'ocolus-blog-main'   => array(
                    'src'     => get_theme_file_uri('/assets/css/blog-main.min.css'),
                    'deps'    => array(),
                    'version' => OCOLUS,
                    'has_rtl' => false,
                ),
                'ocolus-backend'     => array(
                    'src'     => get_theme_file_uri('/assets/css/backend.min.css'),
                    'deps'    => array(),
                    'version' => OCOLUS,
                    'has_rtl' => false,
                ),
                'icomoon'            => array(
                    'src'     => get_theme_file_uri('/assets/fonts/icomoon/style.min.css'),
                    'deps'    => array(),
                    'version' => OCOLUS,
                    'has_rtl' => false,
                ),
                'font-awesome'       => array(
                    'src'     => get_theme_file_uri('/assets/css/fontawesome.min.css'),
                    'deps'    => array(),
                    'version' => '4.7.0',
                    'has_rtl' => false,
                ),
                'fullpage'           => array(
                    'src'     => get_theme_file_uri('/assets/vendor/fullpage/fullpage.min.css'),
                    'deps'    => array(),
                    'version' => '3.0.4',
                    'has_rtl' => false,
                ),
                'scrollbar'          => array(
                    'src'     => get_theme_file_uri('/assets/vendor/scrollbar/scrollbar.min.css'),
                    'deps'    => array(),
                    'version' => '0.2.10',
                    'has_rtl' => false,
                ),
                'magnific-popup'     => array(
                    'src'     => get_theme_file_uri('/assets/vendor/magnific-popup/magnific-popup.min.css'),
                    'deps'    => array(),
                    'version' => '1.1.0',
                    'has_rtl' => false,
                ),
            );
            foreach ($register_styles as $name => $props) {
                self::register_style($name, $props['src'], $props['deps'], $props['version'], 'all', $props['has_rtl']);
            }
        }

        /**
         * Register/queue backend scripts.
         */
        public static function admin_scripts($hook_suffix)
        {
            self::register_styles();
            // CSS Styles.
            self::enqueue_style('icomoon');
            self::enqueue_style('font-awesome');
            if (($hook_suffix === 'post-new.php' || $hook_suffix === 'post.php')) {
                self::enqueue_style('ocolus-backend');
            }
            self::enqueue_script('ocolus-backend');
        }

        public static function dequeue_scripts()
        {
            global $post;
            $disable_zoom = ocolus_get_option('disable_zoom');
            /* DEQUEUE SCRIPTS - OPTIMIZER */
            if (is_a($post, 'WP_Post') && !has_shortcode($post->post_content, 'contact-form-7')) {
                wp_dequeue_style('contact-form-7');
                wp_dequeue_script('contact-form-7');
            }
            /* AWESOME REV */
            if (class_exists('RevSliderFront')) {
                remove_action('wp_footer', array('RevSliderFront', 'load_icon_fonts'));
            }
            /* CHECK IS MOBILE */
            if (ocolus_is_mobile() && class_exists('WooCommerce') && !is_product()) {
                wp_dequeue_style('product-attributes-swatches');
                wp_dequeue_script('product-attributes-swatches');
            }
            /* WOOCOMMERCE */
            if (class_exists('WooCommerce')) {
                if ($disable_zoom == 1) {
                    wp_dequeue_script('zoom');
                }
                if (class_exists('YITH_WCQV_Frontend')) {
                    wp_dequeue_style('yith-quick-view');
                }
                if (defined('YITH_WCWL')) {
                    $wishlist_page_id = yith_wcwl_object_id(get_option('yith_wcwl_wishlist_page_id'));
                    if (!is_page($wishlist_page_id)) {
                        wp_dequeue_script('prettyPhoto');
                        wp_dequeue_script('jquery-selectBox');
                        wp_dequeue_style('woocommerce_prettyPhoto_css');
                        wp_dequeue_style('jquery-selectBox');
                        wp_dequeue_style('yith-wcwl-main');
                        wp_dequeue_style('yith-wcwl-user-main');
                    }
                    wp_dequeue_style('yith-wcwl-font-awesome');
                }
                /* PLUGIN GIFT */
                if (class_exists('Woocommerce_Multiple_Free_Gift')) {
                    $free_gift = false;
                    if (is_product()) {
                        $free_gift = get_post_meta($post->ID, '_wfg_single_gift_products', true);
                    }
                    if (!$free_gift) {
                        wp_dequeue_style('wfg-core-styles');
                        wp_dequeue_style('wfg-styles');
                        wp_dequeue_script('wfg-scripts');
                    }
                }
                /* PLUGIN SIZE CHART */
                if (class_exists('Size_Chart_For_Woocommerce')) {
                    $size_chart = false;
                    if (is_product()) {
                        $size_chart = get_post_meta($post->ID, 'prod-chart', true);
                    }
                    if (!$size_chart) {
                        wp_dequeue_style('size-chart-for-woocommerce');
                        wp_dequeue_style('size-chart-for-woocommerce-select2');
                        wp_dequeue_style('size-chart-for-woocommerce-jquery-modal');
                        wp_dequeue_style('size-chart-for-woocommerce-jquery-modal-default-theme');
                        wp_dequeue_script('size-chart-for-woocommerce');
                        wp_dequeue_script('size-chart-for-woocommerce-jquery-select2');
                        wp_dequeue_script('size-chart-for-woocommerce-jquery-modal');
                        wp_dequeue_script('size-chart-for-woocommerce-jquery-editable-js');
                        wp_dequeue_script('size-chart-for-woocommerce-jquery-modal-default-theme');
                    }
                }
                if (class_exists('Vc_Manager')) {
                    wp_dequeue_script('vc_woocommerce-add-to-cart-js');
                }
            }
        }

        /**
         * Register/queue frontend scripts.
         */
        public static function load_scripts()
        {
            self::register_scripts();
            self::register_styles();
            // Global frontend scripts.
            if (is_page_template('templates/fullpage.php')) {
                self::enqueue_style('fullpage');
                self::enqueue_script('fullpage');
                // Add inline script
                wp_add_inline_script('fullpage', '
				jQuery(document).ready(function ($) {
					if ( $.fn.fullpage ) {
						$("#fullpage").fullpage({
							sectionSelector: ".vc_section",
							navigation: true,
							resize: true,
							responsiveWidth: 1200
						});
					}
				});'
                );
            }
            if (!ocolus_is_mobile()) {
                self::enqueue_style('scrollbar');
                self::enqueue_script('scrollbar');
            }
            if (is_singular() && comments_open() && get_option('thread_comments')) {
                wp_enqueue_script('comment-reply');
            }
            self::enqueue_script('ocolus-main');
            // Add inline script
            $ace_script = ocolus_get_option('ace_script', '');
            if ($ace_script) {
                wp_add_inline_script('ocolus-main', $ace_script);
            }
            // CSS Styles.
            $enqueue_styles = self::get_styles();
            if (!empty($enqueue_styles)) {
                foreach ($enqueue_styles as $handle => $args) {
                    if (!isset($args['has_rtl'])) {
                        $args['has_rtl'] = false;
                    }
                    self::enqueue_style($handle, $args['src'], $args['deps'], $args['version'], $args['media'], $args['has_rtl']);
                }
            }
            // Optimizer scripts
            self::dequeue_scripts();
        }

        /**
         * Localize a Ocolus script once.
         *
         * @since 2.3.0 this needs less wp_script_is() calls due to https://core.trac.wordpress.org/ticket/28404 being added in WP 4.0.
         *
         * @param  string  $handle  Script handle the data will be attached to.
         */
        private static function localize_script($handle)
        {
            if (!in_array($handle, self::$wp_localize_scripts, true) && wp_script_is($handle)) {
                $data = self::get_script_data($handle);
                if (!$data) {
                    return;
                }
                $name                        = str_replace('-', '_', $handle).'_params';
                self::$wp_localize_scripts[] = $handle;
                wp_localize_script($handle, $name, apply_filters($name, $data));
            }
        }

        /**
         * Return data for script handles.
         *
         * @param  string  $handle  Script handle the data will be attached to.
         *
         * @return array|bool
         */
        private static function get_script_data($handle)
        {
            switch ($handle) {
                case 'ocolus-main':
                    $params = array(
                        'ajaxurl'          => admin_url('admin-ajax.php'),
                        'security'         => wp_create_nonce('ocolus_ajax_security'),
                        'ocolus_ajax_url'  => Ocolus_Ajax::get_endpoint('%%endpoint%%'),
                        'products_hover'   => ocolus_get_option('products_hover', 'none'),
                        'ajax_add_to_cart' => ocolus_get_option('enable_ajax_add_to_cart'),
                        'nav_next'         => esc_html__('NEXT', 'ocolus'),
                        'nav_prev'         => esc_html__('PREV', 'ocolus'),
                        'growl_notice'     => apply_filters('ocolus_growl_notice_params',
                            array(
                                'added_to_cart_text'     => esc_html__('Product has been added to cart!', 'ocolus'),
                                'added_to_wishlist_text' => get_option('yith_wcwl_product_added_text', esc_html__('Product has been added to wishlist!', 'ocolus')),
                                'wishlist_url'           => function_exists('YITH_WCWL') ? esc_url(YITH_WCWL()->get_wishlist_url()) : '',
                                'browse_wishlist_text'   => get_option('yith_wcwl_browse_wishlist_text', esc_html__('Browse Wishlist', 'ocolus')),
                                'growl_notice_text'      => esc_html__('Notice!', 'ocolus'),
                                'removed_cart_text'      => esc_html__('Product Removed', 'ocolus'),
                                'growl_duration'         => 3000,
                            )
                        ),
                    );
                    break;
                default:
                    $params = false;
            }

            return apply_filters('ocolus_get_script_data', $params, $handle);
        }

        /**
         * Localize scripts only when enqueued.
         */
        public static function localize_printed_scripts()
        {
            foreach (self::$scripts as $handle) {
                self::localize_script($handle);
            }
        }
    }

    Ocolus_Assets::init();
}